create or replace package body tbicds.PCK_PWEVENT is
/* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

/*gets recent clinics used for events*/
procedure GetRecentClinicRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_nRegionID              in number,
      pi_nSiteID                in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(1500) :=
       
        ' select * 
        from utl_clinic nt
        where nt.clinic_id
        in
        (select distinct(TIU_CLINIC_ID) 
        from pat_pw_event t 
        where t.TIU_CLINIC_ID is not null
        and t.region_id = :REGION_ID
        and t.site_id = :SITE_ID)
        and nt.region_id = :REGION_ID2 and nt.site_id = :SITE_ID2
        order by xfer_date desc';
       
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
     open rs 
      for v_vSQL
    using pi_nRegionID,
          pi_nSiteID,
          pi_nRegionID,
          pi_nSiteID;
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;



/*gets recent note titles used for events*/
procedure GetRecentNoteTitleRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_nRegionID              in number,
      pi_nSiteID                in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(1500) :=
       
        ' select nt.*,
          decode(nt.is_consult,1,''Y'',''N'') as consult_yn 
          from utl_note_title nt
          where nt.note_title_tag
          in
          (select distinct(TIU_NOTE_TITLE_IEN) 
          from pat_pw_event t 
          where t.tiu_note_id is not null
          and t.region_id = :REGION_ID
          and t.site_id = :SITE_ID)
          and nt.region_id = :REGION_ID2 and nt.site_id = :SITE_ID2
          order by xfer_date desc';
       
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
     open rs 
      for v_vSQL
    using pi_nRegionID,
          pi_nSiteID,
          pi_nRegionID,
          pi_nSiteID;
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/***
insert modules into the pat pw event module list
****/
procedure InsertPatPWEventModule (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      
      pi_vPatientID             in varchar2,
      pi_nPatPWEventID          in number,
      pi_nMID                   in number,
      
      po_nEventModuleID         out number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is

      v_nIndex number := 0;
      v_vMIDPiece varchar2(4000) := ''; 
      v_vFORPiece varchar2(4000) := '';  
      v_nMID number := 0;   
      v_nModuleFor number := 2;
      
      v_dtOldScheduled date := null;
      v_nPatPWID number := 0;
      v_nDays number := 0;
      v_nEventModuleID number := 0;
   
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    v_nEventModuleID := 0;
    
        
           --get the mid and bump the index
           v_nMID := pi_nMID;
           v_nModuleFor := 3;
           v_nIndex := v_nIndex +1;
           
           select SEQPATPWEVENTMODULEID.Nextval 
           into v_nEventModuleID from dual;
           
           insert into 
           pat_pw_event_module(PATIENT_ID,
                               MID,
                               STATUS,
                               REGION_ID,
                               SITE_ID,
                               DATE_STARTED,
                               DATE_COMPLETED,
                               PAT_PW_ID,
                               INTAKE_ID,
                               MODULE_FOR,
                               SORT_ORDER,
                               PAT_PW_EVENT_ID,
                               DATE_SCHEDULED,
                               PW_EVENT_MODULE_ID)
            values(pi_vPatientID,
                   
                   v_nMID,
                   
                   0,--status
                   
                   (select REGION_ID 
                    from pat_pw_event 
                    where pat_pw_event_id = pi_nPatPWEventID),
                   
                    (select SITE_ID 
                    from pat_pw_event 
                    where pat_pw_event_id = pi_nPatPWEventID),
                    
                    null, --date started
                    
                    null, --date completed
                   
                   (select PAT_PW_ID 
                    from pat_pw_event 
                    where pat_pw_event_id = pi_nPatPWEventID),
                   
                    null, --intake_id
                    
                    v_nModuleFor, --module for
                    
                    (select nvl(max(sort_order), 0) + 1 
                       from pat_pw_event_module 
                      where pat_pw_event_id = pi_nPatPWEventID),
                      
                    pi_nPatPWEventID,
                    
                    sysdate,
                    
                    v_nEventModuleID);
                                    
           commit;    
           
           po_nEventModuleID :=  v_nEventModuleID;                          
 
exception
    when others then
      rollback;
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*
Insert patient pathway event
*/
procedure InsertPatPWEvent (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      
      pi_nRegionID              in number,
      pi_nSiteID                in number,
      
      pi_vPatientID             in varchar2,
      pi_nPatPWID               in number,
   
      pi_vEventTitle            in varchar2,
      pi_dtEventDate            in date,
      
      po_nPatPWEventID          out number,
      
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is
      v_nPatPWEventID number := 0;
   
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    po_nPatPWEventID := 0;
    
    --get a new patient pathway event id
    select SEQPATPWEVENTID.nextval into v_nPatPWEventID from dual;
    
    --insert the event 
    insert into pat_pw_event
                (PATIENT_ID,
                 STATUS,
                 DATE_SCHEDULED,
                 DATE_STARTED,
                 DATE_COMPLETED,
                 REGION_ID,
                 SITE_ID,
                 PAT_PW_ID,
                 EVENT_TITLE,
                 EVENT_DESCR,
                 SORT_ORDER,
                 EVENT_FREQUENCY,
                 PW_EVENT_TYPE_ID,
                 PAT_PW_EVENT_ID,
                 CREATED_BY)
        values(pi_vPatientID,
               0,
               pi_dtEventDate,
               null,
               null,
               pi_nRegionID,
               pi_nSiteID,
               pi_nPatPWID,
               pi_vEventTitle,
               '',
               99,
               null,
               1, --scheduled
               v_nPatPWEventID,
               pi_nUserID);
        
        commit;                  
        
        --return the patpweventid
        po_nPatPWEventID := v_nPatPWEventID;
        
exception
    when others then
      rollback;
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*****
Gets todays pathway events associated with a patient
******/
procedure GetTodaysPatPWEventRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPatPWID               in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(1500) :=
       
        'SELECT t.PATIENT_ID,  
                t.STATUS,
                t.DATE_SCHEDULED,
                t.DATE_STARTED,
                t.DATE_COMPLETED,
                t.REGION_ID,
                t.SITE_ID,
                t.PAT_PW_ID,
                t.event_title,
                t.EVENT_DESCR,
                t.SORT_ORDER,
                t.EVENT_FREQUENCY,
                t.PW_EVENT_TYPE_ID,
                t.PAT_PW_EVENT_ID,
                t.PLAN_PAT_PW_EVENT_ID,
                t.PLAN_TEXT,
                t.NOTE_TEXT,
                t.TIU_DFN,
                t.TIU_DUZ,
                t.TIU_ENCOUNTER_ID,
                t.TIU_CONSULT_IEN,
                t.TIU_COSIGNER_DUZ,
                t.TIU_NOTE_TITLE_IEN,
                t.TIU_NOTE,
                t.TIU_CLINIC_ID,
                t.TIU_DATE_SUBMITTED,
                t.TIU_ENCOUNTER_TYPE,
                t.TIU_NOTE_TITLE,
                t.TIU_LOCATION,
                t.TIU_COSIGNER,
                t.TIU_NOTE_ID,
                t.TIU_CONSULT
                
       FROM tbicds.PAT_PW_EVENT t
       WHERE t.patient_id = :PATIENT_ID 
         and t.pat_pw_id = :PAT_PWID 
         and trunc(t.date_scheduled) = trunc(sysdate) 
         ORDER BY t.DATE_SCHEDULED DESC';
       
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
     open rs 
      for v_vSQL
    using pi_vPatientID,
          pi_nPatPWID;
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*gets the no pathway for a patient*/
procedure GetNoPathwayRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(1024) :=
      'select t.* 
         from PAT_PW t 
        where t.patient_id = :PATIENT_ID 
              and t.PW_TYPE_ID = 3';

begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
    open rs 
    for v_vSQL 
      using pi_vPatientID;
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;


/*gets all open encounters for a patient*/
procedure GetOpenEncountersRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(2048) :=
      'select to_char(t.date_scheduled,''mm/dd/yyyy HH:MI:SS'') as event_date,
              t.date_completed, t.patient_id, t.tiu_note_title,
              t.pat_pw_event_id,
              t.status,
              t.date_scheduled,
              t.pat_pw_id,
              (select name 
                 from app_user u
              where u.fx_user_id = t.created_by)  as PROVIDER_NAME
              
         from PAT_PW_EVENT t, PAT_PW p
        where t.pat_pw_id = p.pat_pw_id 
              and t.date_completed is null 
              and t.patient_id = :PATIENT_ID 
              and trunc(t.date_scheduled) <= trunc(sysdate) 
              and (( (t.tiu_note_title is not null)
                 or (t.pat_pw_event_id in (select pat_pw_event_id 
                                          from pat_pw_event_module t2 
                                          where t2.date_completed is not null))
                 or (t.note_text is not null))
                 
                 or (t.status = 0 and p.pw_type_id = 3)
                 )
                 
              order by t.date_scheduled desc';

begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
    open rs 
    for v_vSQL 
      using pi_vPatientID;
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;


/*updates the intrument date*/
procedure UpdateInstrumentDate (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      
      pi_vPatientID             in varchar2,
      pi_nIntakeID              in number,
      pi_dtDate                 in date,
                      
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is

     v_vSQL long :=
      'UPDATE tbicds.PAT_PW_EVENT_MODULE  
       set DATE_STARTED = :DATE_STARTED,
           DATE_COMPLETED = :DATE_COMPLETED 
       where PATIENT_ID = :PATIENT_ID
       and INTAKE_ID = :INTAKE_ID';
       
     v_vSQL2 long :=
      'UPDATE tbicds.DATA_INTAKE  
       set START_DATE = :DATE_STARTED,
           COMPLETE_DATE = :DATE_COMPLETED 
       where PATIENT_ID = :PATIENT_ID
       and INTAKE_ID = :INTAKE_ID';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --execute the sql
    execute immediate v_vSQL 
                using pi_dtDate,
                      pi_dtDate,
                      pi_vPatientID,
                      pi_nIntakeID
                      ;
    
    commit;
    
    --execute the sql setting the modules to completed
    execute immediate v_vSQL2 
                using pi_dtDate,
                      pi_dtDate,
                      pi_vPatientID,
                      pi_nIntakeID
                      ;
    
    commit;
    
exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*get the most recent medical history record*/
procedure GetMedicalHxRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(4000) :=
      'select t.patient_id,
              t.region_id,
              t.site_id,
              t.pat_pw_id,
              t.pat_pw_event_id,
              t.med_hx_date,
              t.fx_user_id
         from patient_medical_hx t
        where t.patient_id = :PATIENT_ID
          and t.med_hx_date = (select max(med_hx_date) 
                                 from patient_medical_hx 
                                where patient_id = :PATIENT_ID)';

begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
     open rs 
      for v_vSQL
    using pi_vPatientID,
          pi_vPatientID;
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;


/*insert a new medical history record*/
procedure InsertMedicalHx ( pi_vSessionID           in varchar2,
                            pi_vSessionClientIP     in varchar2,
                            pi_nUserID              in number,
                                
                            pi_nRegionID            in number,
                            pi_nSiteID              in number,
                            pi_vPatientID           in varchar2,
                            pi_nPatPWID             in number,
                            pi_nPatPWEventID        in number,
                            
                            po_nStatusCode           out number,
                            po_vStatusComment        out varchar2
                          )
is
    v_nPWEventTypeID number := 0;
    v_nPatPWEventID number := 0;
    v_nSortOrder number := 0;
    
    v_vSQL long :=

        'insert into tbicds.patient_medical_hx( PATIENT_ID,
                                                REGION_ID,
                                                SITE_ID,
                                                PAT_PW_ID,
                                                PAT_PW_EVENT_ID,
                                                MED_HX_DATE,
                                                FX_USER_ID
                                               )
          values(:PATEINT_ID,
                 :REGION_ID,
                 :SITE_ID,
                 :PAT_PW_ID,
                 :PAT_PW_EVENT_ID,
                 :MED_HX_DATE,
                 :FX_USER_ID             
                 )';
begin
  
    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --get a new id
    --select SEQPATPWEVENTID.nextval into v_nPatPWEventID from dual;
    
    --execute the SQL
    execute immediate v_vSQL
    using pi_vPatientID,
          pi_nRegionID,
          pi_nSiteID,
          pi_nPatPWID,
          pi_nPatPWEventID,
          sysdate,
          pi_nUserID;  
           
    commit;

exception
    when others then
      rollback;
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*get the most recent medical history intake_id*/ 
procedure GetMedicalHxIntakeID( pi_vSessionID       in varchar2,
                                pi_vSessionClientIP in varchar2,
                                pi_nUserID          in number,   
                                pi_vPatientID       in varchar2,
                                po_nIntakeID        out number,   
                                po_nStatusCode      out number,
                                po_vStatusComment   out varchar2
                              )
is

  v_nIntakeID number := 0;

begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    -- get the latest intake_id for the Past Medical History module instrument
    begin
      select max(intake_id)
        into v_nIntakeID
        from data_intake_responses
       where patient_id = pi_vPatientID
         and mid = 3000; -- Past Medical History module instrument
    exception
      when others then
        v_nIntakeID := 0;
    end;
    
    --Return the Intake ID
    po_nIntakeID := v_nIntakeID;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;
                               
/*write note details to tbi cds after commited to MDWS*/
procedure WriteNote (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      
      pi_vPatientID             in varchar2,
      pi_nPatPWEventID          in number,
      
      pi_vDFN                   in varchar2,
      pi_vDUZ                   in varchar2,
      pi_vEncounterID           in varchar2,
      pi_vConsultIEN            in varchar2,
      pi_vCosignerDUZ           in varchar2,
      pi_vNoteTitleIEN          in varchar2,
      pi_vTIUNoteID             in varchar2,
      pi_vTIUNote               in clob,
      pi_nClinicID              in number,
      pi_nEncounterType         in number,
      pi_vNoteTitle             in varchar2,
      pi_vLocation              in varchar2,
      pi_vCosigner              in varchar2,
      pi_vConsult               in varchar2,
      pi_vTBINote               in clob,
                      
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is

     v_vSQL long :=
      'UPDATE tbicds.pat_pw_event  
       set TIU_DFN = :TIU_DFN,
           TIU_DUZ = :TIU_DUZ,
           TIU_ENCOUNTER_ID = :TIU_ENCOUNTER_ID,
           TIU_CONSULT_IEN = :TIU_CONSULT_IEN,
           TIU_COSIGNER_DUZ = :TIU_COSIGNER_DUZ,
           TIU_NOTE_TITLE_IEN = :TIU_NOTE_TITLE_IEN,
           TIU_NOTE_ID = :TIU_NOTE_ID, 
           TIU_NOTE = :TIU_NOTE,
           TIU_CLINIC_ID = :TIU_CLINIC_ID,
           TIU_DATE_SUBMITTED = :TIU_DATE_SUBMITTED,
           TIU_ENCOUNTER_TYPE = :TIU_ENCOUNTER_TYPE,
           TIU_NOTE_TITLE = :TIU_NOTE_TITLE,
           TIU_LOCATION = :TIU_LOCATION,
           TIU_COSIGNER = :TIU_COSIGNER,
           TIU_CONSULT = :TIU_CONSULT,
           NOTE_TEXT = :NOTE_TEXT,
           STATUS = 2,
           DATE_COMPLETED = :DATE_COMPLETED 
       where PATIENT_ID = :PATIENT_ID
       and PAT_PW_EVENT_ID = :PAT_PW_EVENT_ID';
       
     v_vSQL2 long :=
      'UPDATE tbicds.PAT_PW_EVENT_MODULE  
       set STATUS = 2
       where STATUS = 1  
       AND PATIENT_ID = :PATIENT_ID
       AND PAT_PW_EVENT_ID = :PAT_PW_EVENT_ID';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --execute the sql
    execute immediate v_vSQL 
                using pi_vDFN,
                      pi_vDUZ,
                      pi_vEncounterID,
                      pi_vConsultIEN,
                      pi_vCosignerDUZ,
                      pi_vNoteTitleIEN,
                      pi_vTIUNoteID,
                      pi_vTIUNote,
                      pi_nClinicID,
                      sysdate,
                      pi_nEncounterType,
                      pi_vNoteTitle,
                      pi_vLocation,
                      pi_vCosigner,
                      pi_vConsult,
                      pi_vTBINote,
                      sysdate,
                      pi_vPatientID,
                      pi_nPatPWEventID
                      ;
    
    commit;
    
    --execute the sql setting the modules to completed
    execute immediate v_vSQL2 
                using pi_vPatientID,
                      pi_nPatPWEventID;
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*save note progress, used while the note is being written but before 
it is commited*/
procedure SaveNoteProgress (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      
      pi_vPatientID             in varchar2,
      pi_nPatPWEventID          in number,
      
      pi_nEncounterType         in number,
      pi_vNoteTitle             in varchar2,
      pi_vLocation              in varchar2,
      pi_vCosigner              in varchar2,
      pi_vConsult               in varchar2,
      pi_vNote                  in clob,
                      
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2
      )
is

     v_vSQL long :=
      'UPDATE tbicds.pat_pw_event  
       set TIU_ENCOUNTER_TYPE = :TIU_ENCOUNTER_TYPE,
           TIU_NOTE_TITLE = :TIU_NOTE_TITLE,
           TIU_LOCATION = :TIU_LOCATION,
           TIU_COSIGNER = :TIU_COSIGNER,
           TIU_CONSULT = :TIU_CONSULT,
           NOTE_TEXT = :NOTE_TEXT
       where PATIENT_ID = :PATIENT_ID
       and PAT_PW_EVENT_ID = :PAT_PW_EVENT_ID';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --execute the sql
    execute immediate v_vSQL 
                using pi_nEncounterType,
                      pi_vNoteTitle,
                      pi_vLocation,
                      pi_vCosigner,
                      pi_vConsult,
                      pi_vNote,
                      pi_vPatientID,
                      pi_nPatPWEventID
                      ;
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/***
Get all outcome measures associated with an event
***/
procedure GetOutcomeMeasuresRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPatPWEventID          in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL long :=
      'select t.patient_id, 
       t.pat_pw_id, 
       t.pat_pw_event_id, 
       t.pw_event_module_id, 
       t.mid,
       t.intake_id,
       t2.module,
              
       (select count(*)  
          from data_intake_responses
         where intake_id = t.intake_id
           and mid = 4042 
           and tid=1
           and qid=1
           and rid=21) as is_goal,
                                       
       (select tdi.complete_date  
          from data_intake tdi
          where tdi.patient_id = t.patient_id
          and tdi.intake_id = t.intake_id
          and tdi.patient_id = t.patient_id 
          and tdi.mid = t.mid 
          and rownum=1) as current_score_date,
          
       (select tds.score   
          from data_intake_score tds
          where tds.patient_id = t.patient_id
          and tds.intake_id = t.intake_id
          and tds.patient_id = t.patient_id
          and tds.mid = t.mid 
          and tds.series = 0
          and rownum=1) as current_score,
          
         (select tds.interpretation   
          from data_intake_score tds
          where tds.patient_id = t.patient_id
          and tds.intake_id = t.intake_id
          and tds.patient_id = t.patient_id 
          and tds.mid = t.mid 
          and tds.series = 0
          and rownum=1) as current_interpretation,
          
         (select pe.event_entry_type_id 
          from pat_pw_event_entry pe,
               stat_event_status es
          where pe.event_status_id = es.event_status_id
            and pe.patient_id = t.patient_id
            and pe.pw_event_module_id = t.pw_event_module_id
            and pe.event_date = 
             (select max(event_date) 
                from pat_pw_event_entry pe2
               where pe2.pw_event_module_id = t.pw_event_module_id)
            and rownum=1) as event_entry_type_id,
            
         (select es.event_status 
          from pat_pw_event_entry pe,
               stat_event_status es
          where pe.event_status_id = es.event_status_id
            and pe.patient_id = t.patient_id
            and pe.pw_event_module_id = t.pw_event_module_id
            and pe.event_date = 
             (select max(event_date) 
                from pat_pw_event_entry pe2
               where pe2.pw_event_module_id = t.pw_event_module_id)
            and rownum=1) as current_status,
            
          (select pe.event_status_id  
          from pat_pw_event_entry pe,
               stat_event_status es
          where pe.event_status_id = es.event_status_id
            and pe.patient_id = t.patient_id
            and pe.pw_event_module_id = t.pw_event_module_id
            and pe.event_date = 
             (select max(event_date) 
                from pat_pw_event_entry pe2
               where pe2.pw_event_module_id = t.pw_event_module_id)
            and rownum=1) as current_status_id,
            
          (select es.event_severity  
          from pat_pw_event_entry pe,
               stat_event_severity es
          where pe.event_severity_id = es.event_severity_id 
            and pe.patient_id = t.patient_id
            and pe.pw_event_module_id = t.pw_event_module_id
            and pe.event_date = 
             (select max(event_date) 
                from pat_pw_event_entry pe2
               where pe2.pw_event_module_id = t.pw_event_module_id)
            and rownum=1) as current_severity,
            
          (select es.event_severity_id  
          from pat_pw_event_entry pe,
               stat_event_severity es
          where pe.event_severity_id = es.event_severity_id 
            and pe.patient_id = t.patient_id
            and pe.pw_event_module_id = t.pw_event_module_id
            and pe.event_date = 
             (select max(event_date) 
                from pat_pw_event_entry pe2
               where pe2.pw_event_module_id = t.pw_event_module_id)
            and rownum=1) as current_severity_id,
                             
         (select count(*) 
          from pat_pw_event_entry pe,
                stat_event_status es
          where pe.event_status_id = es.event_status_id
            and pe.patient_id = t.patient_id
            and pe.pw_event_module_id = t.pw_event_module_id
            and rownum = 1) as current_reviewed,
            
       (select tdi.complete_date - 10  
          from data_intake tdi
          where tdi.patient_id = t.patient_id
          and tdi.intake_id = t.intake_id
          and tdi.patient_id = t.patient_id 
          and tdi.mid = t.mid 
          and rownum=1) as prev_score_date,
          
          PCK_PWEVENT.GetPrevIntakeScore(t.mid, t.patient_id, t.intake_id) as prev_score,
          
         (select tds.interpretation || ''PREV''   
          from data_intake_score tds
          where tds.patient_id = t.patient_id
          and tds.intake_id = t.intake_id 
          and tds.patient_id = t.patient_id 
          and tds.mid = t.mid 
          and tds.series = 0
          and rownum=1) as prev_interpretation,
          
          (select nvl(tt.status, 0) 
             from pat_pw_event tt 
           where tt.pat_pw_event_id = :PAT_PW_EVENT_ID) 
           as event_status
          
          from PAT_PW_EVENT_MODULE t,
               intake_module t2
          where t.patient_id = :PATIENT_ID
            and t.pat_pw_event_id = :PAT_PW_EVENT_ID
            and t2.mid = t.mid 
            and nvl(t.intake_id,-1) > 0 
       order by t2.module, t.date_scheduled';

begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
     open rs 
      for v_vSQL
    using pi_nPatPWEventID,
          pi_vPatientID,
          pi_nPatPWEventID;
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/***
Get all scores for an intake
***/
procedure GetIntakeScoreRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nIntakeID              in number,
      pi_nMID                   in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(4000) :=
      'select t.patient_id,
              t.mid,
              t.intake_id,
              t.series,
              t.score,
              PCK_PWEVENT.GetPrevIntakeSubscaleScore(t.mid, t.patient_id, t.intake_id, t.series) as PREV_SCORE,
              t.interpretation,
              t.description
         from data_intake_score t
        where t.patient_id = :PATIENT_ID
          and t.intake_id = :INTAKE_ID
          and t.MID = :MID 
     order by t.series asc';

begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
     open rs 
      for v_vSQL
    using pi_vPatientID,
          pi_nIntakeID,
          pi_nMID;
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/******
get all event entries for an event
******/  
procedure GetEventEntryRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPatPWEventID          in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(4000) :=
      'SELECT t.PAT_PW_EVENT_ENTRY_ID,
              t.PATIENT_ID,
              t.PAT_PW_ID,
              t.PAT_PW_EVENT_ID,
              t.EVENT_ENTRY_TYPE_ID,
              t.EVENT_COMMENT,
              t.FX_USER_ID,
              t.EVENT_DATE,
              t.EVENT_STATUS_ID,
              t.EVENT_SEVERITY_ID,
              t.PW_EVENT_MODULE_ID,
              t2.NAME as USER_NAME,
              t2.TITLE as USER_TITLE,
              t3.EVENT_ENTRY_TYPE,
              
              (select tim.module from 
               intake_module tim,
               pat_pw_event_module tpe 
               where tpe.PW_EVENT_MODULE_ID = t.PW_EVENT_MODULE_ID 
               and tim.mid = tpe.mid) as module, 
              
              (select count(*) from
                tbicds.pat_pw_event_entry
                where PAT_PW_EVENT_ENTRY_ID = t.PAT_PW_EVENT_ENTRY_ID
                and fx_user_id = :FX_USER_ID) as allow_edit,
              (select nvl(tt.status,0) from
                pat_pw_event tt
                where tt.pat_pw_event_id = :PAT_PW_EVENT_ID) as event_status
                
       FROM tbicds.pat_pw_event_entry t,
            tbicds.app_user t2,
            tbicds.stat_event_entry_type t3
       where t.pat_pw_event_id = :PAT_PW_EVENT_ID 
       and t.patient_id = :PATIENT_ID
       and t2.fx_user_id = t.fx_user_id 
       and t3.event_entry_type_id = t.event_entry_type_id 
       order by event_date desc';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
     open rs 
      for v_vSQL
    using pi_nUserID,
          pi_nPatPWEventID, 
          pi_nPatPWEventID,
          pi_vPatientID;
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;


/******
get event entry types Subjective, Objective, Assessment, Plan etc...
******/  
procedure GetEventEntryTypeRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(500) :=
      'SELECT t.*
       FROM tbicds.STAT_EVENT_ENTRY_TYPE t 
       order by sort_order asc';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
    open rs 
    for v_vSQL;
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/******
get event statuses Better, Same, Worse etc...
******/  
procedure GetEventStatusRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(500) :=
      'SELECT t.*
       FROM tbicds.stat_event_status t 
       order by sort_order asc';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
    open rs 
    for v_vSQL;
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/******
get event severities: mild, moderate, severe etc...
******/  
procedure GetEventSeverityRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(500) :=
      'SELECT t.*
       FROM tbicds.stat_event_severity t 
       order by sort_order asc';
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    --open the result set
    open rs 
    for v_vSQL;
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*
insert a new pathway event
*/
procedure InsertPWEvent ( pi_vSessionID           in varchar2,
                          pi_vSessionClientIP     in varchar2,
                          pi_nUserID              in number,
                                
                          pi_nRegionID            in number,
                          pi_nSiteID              in number,
                          pi_vPatientID           in varchar2,
                          pi_nPatPWID             in number,
                          pi_vEventTitle          in varchar2,
                          pi_dtEventDate          in varchar2,
                          pi_nRelatedEventID      in number,
                                
                          po_nPatPWEventID           out number,
                          po_nStatusCode           out number,
                          po_vStatusComment        out varchar2
                          )
is
    v_nPWEventTypeID number := 0;
    v_nPatPWEventID number := 0;
    v_nSortOrder number := 0;
    
    v_vSQL long :=

        'insert into tbicds.pat_pw_event( PATIENT_ID,
                                          STATUS,
                                          DATE_SCHEDULED,
                                          REGION_ID,
                                          SITE_ID,
                                          PAT_PW_ID,
                                          EVENT_TITLE,
                                          SORT_ORDER,
                                          EVENT_FREQUENCY,
                                          PW_EVENT_TYPE_ID,
                                          PAT_PW_EVENT_ID,
                                          MERGED
                                         )
          values(:PATEINT_ID,
                 :STATUS,
                 :DATE_SCHEDULED,
                 :REGION_ID,
                 :SITE_ID,
                 :PAT_PW_ID,
                 :EVENT_TITLE,
                 :SORT_ORDER,
                 :EVENT_FREQUENCY,
                 :PW_EVENT_TYPE_ID,
                 :PAT_PW_EVENT_ID,
                 :MERGED                 
                 )';
begin
  
    po_nStatusCode    := 0;
    po_vStatusComment := '';
    po_nPatPWEventID := 0;
    
    --get a new id
    select SEQPATPWEVENTID.nextval into v_nPatPWEventID from dual;
    
    --get event type
    select t.pw_event_type_id, 
           t.sort_order into 
           v_nPWEventTypeID, v_nSortOrder 
    from pat_pw_event t
    where t.pat_pw_event_id = pi_nRelatedEventID
    and t.patient_id = pi_vPatientID;
  
    --bump sort order for all records after related record
    update pat_pw_event tt
    set tt.sort_order = sort_order + 1
    where tt.patient_id = pi_vPatientID 
    and tt.pat_pw_event_id = pi_nRelatedEventID
    and tt.sort_order > v_nSortOrder;
    
    --execute the SQL
    execute immediate v_vSQL
    using pi_vPatientID,
          0,
          pi_dtEventDate,
          pi_nRegionID,
          pi_nSiteID,
          pi_nPatPWID,
          pi_vEventTitle,
          v_nSortOrder + 1,
          0,
          v_nPWEventTypeID,
          v_nPatPWEventID,
          0;  
           
    commit;

    --return the id
    po_nPatPWEventID := v_nPatPWEventID;
   

exception
    when others then
      rollback;
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/******
insert an event entry 
******/  
procedure InsertPWEventEntry (  pi_vSessionID             in varchar2,
                                pi_vSessionClientIP       in varchar2,
                                pi_nUserID                in number,
                                
                                pi_vPatientID             in varchar2,
                                pi_nPatPWEventID          in number,
                                pi_nPWEventModuleID       in number,
                                pi_nEventEntryTypeID      in number,
                                pi_dtEventDate            in date,
                                pi_vEventEntryComment     in varchar2,
                                pi_nEventStatusID         in number,
                                pi_nEventSeverityID       in number,
                                
                                po_nEventEntryID          out number,
                                
                                po_nStatusCode           out number,
                                po_vStatusComment        out varchar2
                             )
is

     v_vSQL long :=

        'insert into tbicds.pat_pw_event_entry(PAT_PW_EVENT_ENTRY_ID,
                                              PATIENT_ID,
                                              PAT_PW_ID,
                                              PAT_PW_EVENT_ID,
                                              EVENT_ENTRY_TYPE_ID,
                                              EVENT_COMMENT,
                                              FX_USER_ID,
                                              EVENT_DATE,
                                              EVENT_STATUS_ID,
                                              EVENT_SEVERITY_ID,
                                              PW_EVENT_MODULE_ID)
          values(:PAT_PW_EVENT_ENTRY_ID,
                 :PATIENT_ID,
                 :PAT_PW_ID,
                 :PAT_PW_EVENT_ID,
                 :EVENT_ENTRY_TYPE_ID,
                 :EVENT_COMMENT,
                 :FX_USER_ID,
                 :EVENT_DATE,
                 :EVENT_STATUS_ID,
                 :EVENT_SEVERITY_ID,
                 :PW_EVENT_MODULE_ID
                 )';
                 
     
      v_nEntryID number := 0;
      v_nPatPWID number := 0;
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
  
    po_nEventEntryID := 0;
    v_nEntryID := 0;
    
    --get a new id
    select seqpatpwentryid.nextval into v_nEntryID from dual;
    
    --get pat pw id
    select t.pat_pw_id into v_nPatPWID 
    from pat_pw_event t
    where t.pat_pw_event_id = pi_nPatPWEventID;
  
    --execute the SQL
    execute immediate v_vSQL 
    using v_nEntryID,
          pi_vPatientID,
          v_nPatPWID,
          pi_nPatPWEventID,
          pi_nEventEntryTypeID,
          pi_vEventEntryComment,
          pi_nUserID,
          pi_dtEventDate,
          pi_nEventStatusID,
          pi_nEventSeverityID,
          pi_nPWEventModuleID;
           
    commit;

    --return the id
    po_nEventEntryID := v_nEntryID;
   

exception
    when others then
      rollback;
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;


/******
update an event entry 
******/  
procedure UpdatePWEventEntry (  pi_vSessionID             in varchar2,
                                pi_vSessionClientIP       in varchar2,
                                pi_nUserID                in number,
                                
                                pi_nEventEntryID          in number,
                                pi_nPWEventModuleID       in number,
                                pi_dtEventDate            in date,
                                pi_vEventEntryComment     in varchar2,
                                pi_nEventStatusID         in number,
                                pi_nEventSeverityID       in number,
                             
                                po_nStatusCode           out number,
                                po_vStatusComment        out varchar2
                             )
is

     v_vSQL long :=

        'update tbicds.pat_pw_event_entry
         set event_comment = :EVENT_COMMENT,
         event_date = :EVENT_DATE,
         event_status_id = :EVENT_STATUS_ID,
         event_severity_ID = :EVENT_SEVERITY_ID
         where PAT_PW_EVENT_ENTRY_ID = :PAT_PW_EVENT_ENTRY_ID
         ';
     
      v_nEventEntryTypeID number := 0;
      v_nEventID number :=0;
     
 begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';  
  
    --get the type of entry
    select t.event_entry_type_id into v_nEventEntryTypeID
    from tbicds.pat_pw_event_entry t
    where t.pat_pw_event_entry_id = pi_nEventEntryID;
    
    --get the event id
    select t.pat_pw_event_id into v_nEventID
    from tbicds.pat_pw_event_entry t
    where t.pat_pw_event_entry_id = pi_nEventEntryID;    
 
    --execute the SQL
    execute immediate v_vSQL 
    using pi_vEventEntryComment,
          pi_dtEventDate,
          pi_nEventStatusID,
          pi_nEventSeverityID,
          pi_nEventEntryID;
           
    commit;
    
exception
    when others then
      rollback;
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/*
update the plan portion of the event
*/
procedure UpdatePWEventPlan (  pi_vSessionID             in varchar2,
                               pi_vSessionClientIP       in varchar2,
                               pi_nUserID                in number,
                               
                               pi_nPatPWEventID          in number,
                               pi_nPatPWPlanEventID      in number,
                               pi_vPlanText              in clob,
                               
                               po_nStatusCode           out number,
                               po_vStatusComment        out varchar2
                             )
is

     v_vSQL long :=

        'update tbicds.pat_pw_event
         set plan_text = :PLAN_TEXT,
         PLAN_PAT_PW_EVENT_ID = :PLAN_PAT_PW_EVENT_ID
         where PAT_PW_EVENT_ID = :PAT_PW_EVENT_ID
         ';
     
 begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';  
  
    --execute the SQL
    execute immediate v_vSQL 
    using pi_vPlanText,
          pi_nPatPWPlanEventID,
          pi_nPatPWEventID;
           
    commit;
    
exception
    when others then
      rollback;
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/***
delete an event entry
***/
procedure DeletePWEventEntry (  pi_vSessionID             in varchar2,
                                pi_vSessionClientIP       in varchar2,
                                pi_nUserID                in number,
                                
                                pi_nEventEntryID          in number,
                                po_nStatusCode           out number,
                                po_vStatusComment        out varchar2
                             )
is

     v_vSQL long :=

        'delete from tbicds.pat_pw_event_entry
         where PAT_PW_EVENT_ENTRY_ID = :PAT_PW_EVENT_ENTRY_ID
         ';
     
 begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';  
  
    --execute the SQL
    execute immediate v_vSQL 
    using pi_nEventEntryID;
           
    commit;
    
exception
    when others then
      rollback;
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;

/***
Gets the previous score value for a mid
***/
function GetPrevIntakeScore(pi_nMID in number,
                            pi_vPatientID in varchar2,
                            pi_nIntakeID in number)
  
 return number
 
 is
 
   v_nScore number := 0;
   
 begin
   
 if pi_nMID != 4042 then
   
   select score into v_nScore 
   from 
      (select s1.score   
         from data_intake_score s1,
              data_intake s2
        where s2.patient_id = s1.patient_id
          and s2.intake_id = s1.intake_id
          and s2.mid = s2.mid
            
          and s1.mid = pi_nMID
          and s1.patient_id = pi_vPatientID 
          and s1.series = 0 
            
          and s2.complete_date < 
                     
          (select complete_date   
            from data_intake
           where patient_id = pi_vPatientID
             and intake_id = pi_nIntakeID
             and mid = pi_nMID 
             and rownum = 1 ) 
      order by s2.complete_date desc
      )where rownum = 1; 

else
  
 select score into v_nScore 
   from 
      (select s1.score   
         from data_intake_score s1,
              data_intake s2
        where s2.patient_id = s1.patient_id
          and s2.intake_id = s1.intake_id
          and s2.mid = s2.mid
            
          and s1.mid = pi_nMID
          and s1.patient_id = pi_vPatientID 
          and s1.series = 0 
          
            and ((select rid 
              from data_intake_responses 
             where intake_id = s1.intake_id 
               and tid=1 
               and qid=1 ) != 21)
            
          and s2.complete_date < 
                     
          (select complete_date   
            from data_intake
           where patient_id = pi_vPatientID
             and intake_id = pi_nIntakeID
             and mid = pi_nMID 
             and rownum = 1 ) 
      order by s2.complete_date desc
      )where rownum = 1; 

  end if;

    
    return v_nScore;

exception
    when others then
    return null;  
end;

 /***
 Gets the previous score value for the subscale of a mid
 ***/
 function GetPrevIntakeSubscaleScore(pi_nMID in number,
                                     pi_vPatientID in varchar2,
                                     pi_nIntakeID in number,
                                     pi_nSeries in number)
  
 return number
 
 is
 
   v_nScore number := 0;
   
 begin
   
 if pi_nMID != 4042 then
   
   select s.score into v_nScore
    from data_intake_score s
   where s.patient_id = pi_vPatientID
     and s.mid = pi_nMID
     and s.series = pi_nSeries
     and s.intake_id =
          (select t.intake_id
             from
               (select l.intake_id
                  from data_intake l
                 where l.mid = pi_nMID
                   and l.patient_id = pi_vPatientID 
                   and l.complete_date <              
                         (select complete_date   
                           from data_intake
                          where patient_id = pi_vPatientID
                            and intake_id = pi_nIntakeID
                            and mid = pi_nMID 
                            and rownum = 1) 
                 order by l.complete_date desc) t
           where rownum = 1);
 else
   
 select s.score into v_nScore
    from data_intake_score s
   where s.patient_id = pi_vPatientID
     and s.mid = pi_nMID
     and s.series = pi_nSeries
     and s.intake_id =
          (select t.intake_id
             from
               (select l.intake_id
                  from data_intake l
                 where l.mid = pi_nMID
                   and l.patient_id = pi_vPatientID 
                   
                   and ((select rid 
                  from data_intake_responses 
                 where intake_id = l.intake_id 
                   and tid=1 
                   and qid=1 ) != 21)
               
                   and l.complete_date <              
                         (select complete_date   
                           from data_intake
                          where patient_id = pi_vPatientID
                            and intake_id = pi_nIntakeID
                            and mid = pi_nMID 
                            and rownum = 1) 
                 order by l.complete_date desc) t
           where rownum = 1);
 
 end if;
    
    return v_nScore;

exception
    when others then
    return null;  
end;

/***
 Gets the next event id after an event
***/
function GetNextPatPWEventID(pi_nPatPWEventID in number,
                             pi_vPatientID in varchar2)
  
 return number
 
 is
 
   v_nPatPWID number := 0;
   v_nPatPWEventID number := 0;
   
 begin
   
   --get the patient pathway id
   select pat_pw_id into v_nPatPWID
     from pat_pw_event
    where pat_pw_event_id = pi_nPatPWEventID 
      and patient_id = pi_vPatientID;
   
   --get the next event id after this event id
   --exclude closed events
   select pat_pw_event_id into v_nPatPWEventID 
   from 
      (select t.pat_pw_event_id   
         from pat_pw_event t
        where t.patient_id = pi_vPatientID
          and t.pat_pw_id = v_nPatPWID
          and nvl(t.status,0) != 2  
          and t.date_scheduled > (select date_scheduled 
                                     from pat_pw_event 
                                    where pat_pw_id = v_nPatPWID
                                      and patient_id = pi_vPatientID 
                                      and pat_pw_event_id = pi_nPatPWEventID )
      order by t.date_scheduled asc
      )where rownum = 1; 

    
    return v_nPatPWEventID;

exception
    when others then
    return null;  
end;

/*****
Gets next pathway event associated with a patient
******/
procedure GetNextPatPWEventRS (
      pi_vSessionID             in varchar2,
      pi_vSessionClientIP       in varchar2,
      pi_nUserID                in number,
      pi_vPatientID             in varchar2,
      pi_nPatPWEventID          in number,
      po_nStatusCode            out number,
      po_vStatusComment         out varchar2,
      rs                        out RetRefCursor
      )
is

     v_vSQL varchar2(1500) :=
       
        'SELECT t.PATIENT_ID,	
                t.STATUS,
                t.DATE_SCHEDULED,
                t.DATE_STARTED,
                t.DATE_COMPLETED,
                t.REGION_ID,
                t.SITE_ID,
                t.PAT_PW_ID,
                t.event_title,
                t.EVENT_DESCR,
                t.SORT_ORDER,
                t.EVENT_FREQUENCY,
                t.PW_EVENT_TYPE_ID,
                t.PAT_PW_EVENT_ID

       FROM tbicds.PAT_PW_EVENT t
       WHERE t.patient_id = :PATIENT_ID 
         and t.pat_pw_event_id = :PAT_PW_EVENT_ID 
         ORDER BY t.DATE_SCHEDULED ASC';
         
    v_nNextEventID number := 0;
       
begin

    po_nStatusCode    := 0;
    po_vStatusComment := '';
    
    v_nNextEventID := PCK_PWEVENT.GetNextPatPWEventID(pi_nPatPWEventID,
                                                      pi_vPatientID);
    
    --open the result set
     open rs 
      for v_vSQL
    using pi_vPatientID,
          v_nNextEventID;
    
    commit;

exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '';
      
end;


end PCK_PWEVENT;
/

